package com.agilex.healthcare.veteranappointment.datalayer.careteam;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.TypedQuery;

import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientProvider;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientProviders;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilter;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.healthcare.utility.PatientIdentityHelper;
import com.agilex.healthcare.veteranappointment.domain.PatientProviderWithFacility;

public class CdwPatientCareTeamDataLayer implements ProviderDataLayer {

    @PersistenceContext(unitName="cdwPersistenceUnit")
    protected EntityManager entityManager;

    private static final String WHERE_PATIENT_ID_CARE_TEAM = "FROM PatientCareTeamPO where patientIcn = :patientIcn and sta3n = :siteCode";
    private static final String WHERE_PATIENT_ID_MH_VISITS = "FROM MentalHealthVisitPO where patientICN = :patientIcn and sta3n = :siteCode and visitDateTime >= :startDate and visitDateTime <= :endDate";
    private static final String WHERE_PATIENT_ID_CARE_TEAM_NO_FACILITY = "FROM PatientCareTeamPO where patientIcn = :patientIcn";
    private static final String WHERE_PATIENT_ID_MH_VISITS_NO_FACILITY = "FROM MentalHealthVisitPO where patientICN = :patientIcn and visitDateTime >= :startDate and visitDateTime <= :endDate";

    @Override
    public PatientProviders fetchPrimaryCareProviders(PatientIdentifier patientIdentifier, String siteCode, boolean byFacility) {
        PatientIdentifier truncatedIcn = truncateICN(patientIdentifier);
    	TypedQuery<PatientCareTeamPO> query = null;
    	
    	if(byFacility){
	        query = this.entityManager.createQuery(WHERE_PATIENT_ID_CARE_TEAM, PatientCareTeamPO.class);
	        query.setParameter("siteCode", siteCode);
    	}
    	else{
    		query = this.entityManager.createQuery(WHERE_PATIENT_ID_CARE_TEAM_NO_FACILITY, PatientCareTeamPO.class);	
    	}
    	query.setParameter("patientIcn", truncatedIcn.getUniqueId());
        
        List<PatientCareTeamPO> patientCareTeam = query.getResultList();

        PatientProviders patientProviders = createPatientProvidersFromCareTeam(patientCareTeam);
        resetPatientIdentifiers(patientIdentifier, patientProviders);

        return patientProviders;
    }

    @Override
    public PatientProviders fetchMentalHealthProviders(PatientIdentifier patientIdentifier, DateFilter filter, String siteCode, boolean byFacility) {
    	PatientIdentifier truncatedIcn = truncateICN(patientIdentifier);
    	TypedQuery<MentalHealthVisitPO> query = null;
        if(byFacility){
	    	query = this.entityManager.createQuery(WHERE_PATIENT_ID_MH_VISITS, MentalHealthVisitPO.class);
	        query.setParameter("siteCode", siteCode);
        }
        else{
        	query = this.entityManager.createQuery(WHERE_PATIENT_ID_MH_VISITS_NO_FACILITY, MentalHealthVisitPO.class);
        }
        query.setParameter("patientIcn", truncatedIcn.getUniqueId());
        query.setParameter("startDate", startDateOrDefault(filter));
        query.setParameter("endDate", endDateOrDefault(filter));

        List<MentalHealthVisitPO> visit = query.getResultList();

        PatientProviders patientProviders = createPatientProvidersFromMHVisits(visit);
        resetPatientIdentifiers(patientIdentifier, patientProviders);

        return patientProviders;
    }


    PatientProviders createPatientProvidersFromCareTeam(List<PatientCareTeamPO> careTeam) {
        PatientProviders patientProviders = new PatientProviders();
        ArrayList<String> tempProvider = new ArrayList<String>();
        
        if(NullChecker.isNotNullish(careTeam)) {
            for(PatientCareTeamPO teamMember : careTeam) {
            	 if(tempProvider.contains(teamMember.getStaffName())){
                 	continue;
                 }
            	PatientProviderWithFacility     provider;
                provider = new PatientProviderWithFacility();
                provider.setProviderName(teamMember.getStaffName());
                provider.setPersonClass(teamMember.getPrimaryStandardPosition());
                provider.setVisitDate(teamMember.getStartDate());
                provider.setFacilityCode(teamMember.getInstitutionCode());
                provider.setFacilityName(teamMember.getInstitutionName());
                patientProviders.add(provider);
                tempProvider.add(teamMember.getStaffName());
            }
        }

        return patientProviders;
    }

    PatientProviders createPatientProvidersFromMHVisits(List<MentalHealthVisitPO> mhVisits) {
        PatientProviders patientProviders = new PatientProviders();
        ArrayList<String> tempProvider = new ArrayList<String>();
        
        if(NullChecker.isNotNullish(mhVisits)) {
            for(MentalHealthVisitPO visit : mhVisits) {
                if(tempProvider.contains(visit.getFirstName()+"-"+visit.getLastName())){
                	continue;
                }
                PatientProviderWithFacility     provider;
                provider = new PatientProviderWithFacility();
                provider.setProviderName(visit.getFirstName() + " " + visit.getLastName());
                provider.setPersonClass(visit.getProviderClass());
                provider.setVisitDate(visit.getVisitDateTime());
                provider.setFacilityCode(Integer.toString(visit.getSiteCode()));
                provider.setFacilityName(visit.getInstitutionName());
                patientProviders.add(provider);
                tempProvider.add(visit.getFirstName()+"-"+visit.getLastName());
            }
        }

        return patientProviders;
    }

    void resetPatientIdentifiers(PatientIdentifier patientidentifier, PatientProviders providers) {
        for(PatientProvider provider : providers) {
            provider.setPatientIdentifier(patientidentifier);
        }
    }


    Date startDateOrDefault(DateFilter filter) {
        // If no start date is provided, filter to 12 years of data
        if(filter == null || filter.getStartDate() == null)
            return DateHelper.minusMonths(new Date(), 24);
        return filter.getStartDate();
    }

    Date endDateOrDefault(DateFilter filter) {
        // if no end date is provided, assume vitals until today's date
        if(filter == null || filter.getEndDate() == null)
            return new Date();
        return filter.getEndDate();
    }

    PatientIdentifier truncateICN(PatientIdentifier patientIdentifier) {
        PatientIdentityHelper helper = new PatientIdentityHelper();
        return helper.truncateICN(patientIdentifier);
    }
}
